// ====================================================================
// This file is part of the Endmember Induction Algorithms Toolbox for SCILAB 
// Copyright (C) Grupo de Inteligencia Computacional, Universidad del 
// País Vasco (UPV/EHU), Spain, released under the terms of the GNU 
// General Public License.
//
// Endmember Induction Algorithms Toolbox is free software: you can redistribute 
// it and/or modify it under the terms of the GNU General Public License 
// as published by the Free Software Foundation, either version 3 of the 
// License, or (at your option) any later version.
//
// Endmember Induction Algorithms Toolbox is distributed in the hope that it will
// be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
// of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Endmember Induction Algorithms Toolbox. 
// If not, see <http://www.gnu.org/licenses/>.
// ====================================================================

function [E] = EIA_WM(data)
  //// [E] = EIA_WM(data)
  // 
  // Manuel Grana <manuel.grana[AT]ehu.es>
  // Miguel Angel Veganzones <miguelangel.veganzones[AT]ehu.es>
  // Grupo de Inteligencia Computacional (GIC), Universidad del Pais Vasco /
  // Euskal Herriko Unibertsitatea (UPV/EHU)
  // http://www.ehu.es/computationalintelligence
  // 
  // Copyright (2011) Grupo de Inteligencia Computacional @ Universidad del Pais Vasco, Spain.
  //
  // Ritter's WM endmembers induction algorithm.
  // ------------------------------------------------------------------------------
  // Input:   data      : column data matrix [nvariables x nsamples]
  //
  // Output:  E         : set of induced endmembers [nchannels x p]
  //
  // Bibliographical references:
  // [1] G. X. Ritter y G. Urcid, “A lattice matrix method for hyperspectral image unmixing”, Information Sciences, vol. In Press, Corrected Proof, Oct. 2010.
  
  //// Arguments
  [lhs,rhs]=argn(0);
  if lhs < 1
    error('Insuficient parameters');
  end 
  
  //// Algorithm
  // hyperbox corners
  u = max(data,'c');
  v = min(data,'c');
  // LAMs
  [W,M] = EIA_LAM(data,data);
  // polytope
  [r_w,c_w] = size(W);
  for i=1:c_w
    W(:,i) = W(:,i) + u(i);
    M(:,i) = M(:,i) + v(i);
  end
  // return the set of induced endmembers
  E = zeros(r_w,2*c_w+2);
  E(:,1:c_w) = W;
  E(:,c_w+1:2*c_w) = M;
  E(:,2*c_w+1) = u;
  E(:,2*c_w+2) = v;
    
endfunction
